/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdlib.h>
#include <string.h>

#include "cmp.h"
#include "kalc.h"
#include "mem.h"


int progNesting = 0;      /* The current level of nesting inside programs */
int lastArrowLevel[COMPLEVELS];  /* Levels at which -> was found */
int lastArrowDepth[COMPLEVELS];  /* Correspoding depth */
int nArrows = -1;                /* Marks lastArrow stack level */
static CtrFlow flow[COMPLEVELS]; /* For keeping track of control statements */
int nFlow = -1;                  /* Marks flow structure stack depth */


void
startProgram(void) 
{
  /*
   * This function is called by the presence of <<, the start of
   * program delimiter. It merely marks the stack position.
   */

  stackMarks[nMarks++] = _f_depth();
}


void
endProgram(void)
{
  /*
   * This function is called by the presence of >>, the end of program
   * delimiter. It joins the objects in the stack into a single program
   * object, until the last mark.
   */

  Object theProg;
  Composite *comp = NULL, *lastComp = NULL;

  if (nMarks == 0)
    doError("Syntax", ERR_UNSTARTEDPROG);

  theProg.type = TYPE_PROG;
  nMarks--;

  while (_f_depth() != stackMarks[nMarks]) {
    comp = (Composite *) malloc(sizeof(Composite));
    if (!comp)
      doError("Fatal", ERR_NOTENOUGHMEMORY);    

    comp->obj = *tos;
    comp->prev = NULL;
    comp->next = lastComp;
    if (lastComp)
      lastComp->prev = comp;
    lastComp = comp;
    tos--;
  }

  theProg.value.comp = comp;

  insertNoDup(theProg);
}


void
evalProgram(Composite *comp)
{
  /*
   * This function evaluates the program starting at the given object.
   */

  int memLevelAtEntry = currMemLevel;

  ++progNesting;

  while (comp) {
    evaluateObj(*comp->obj, 0);
    
    if (type(*comp->obj) == TYPE_FUNC) {
      if ((*comp->obj).value.func == f_then) {
	if (flow[nFlow].endValue == 0) {
	  if (flow[nFlow].type == CTR_IF) {
	    skipProgram(&comp, f_else, f_end);
	    if (comp == NULL)
	      doError("then", ERR_SYNTAXERROR);
	    if ((*comp->obj).value.func == f_end)
	      --nFlow;
	  } else { /* CTR_CASE */
	    skipProgram(&comp, f_end, NULL);
	    if (comp == NULL)
	      doError("then", ERR_SYNTAXERROR);
	    flow[nFlow].currValue = 0;
	  }
	}
      } else if ((*comp->obj).value.func == f_else) {
	if (flow[nFlow].endValue == 1) {
	  --nFlow;
	  skipProgram(&comp, f_end, NULL);
	  if (comp == NULL)
	    doError("else", ERR_SYNTAXERROR);
	}
      } else if ((*comp->obj).value.func == f_start
		 || (*comp->obj).value.func == f_while
		 || (*comp->obj).value.func == f_do)
	flow[nFlow].address = comp;
      else if ((*comp->obj).value.func == f_for) {
	Object *obj = (Object *) malloc(sizeof(Object));
	if (!obj)
	  doError("Fatal", ERR_NOTENOUGHMEMORY);
	
	obj->type = TYPE_REAL;
	obj->value.real = flow[nFlow].currValue;
	
	comp = comp->next;
	flow[nFlow].address = comp;
	if (type(*comp->obj) != TYPE_ID && type(*comp->obj) != TYPE_UNQUOTEDID)
	  doError("for", ERR_BADARGUMENTTYPE);
	flow[nFlow].ctrVarName = (*comp->obj).value.str;
	initNextMemoryLevel();
	bindObject(strdup(flow[nFlow].ctrVarName), obj);
      } else if ((*comp->obj).value.func == f_next
		 || (*comp->obj).value.func == f_step) {
	if (flow[nFlow].currValue <= flow[nFlow].endValue)
	  comp = flow[nFlow].address;
	else {
	  if (flow[nFlow--].ctrVarName)
	    clearMemoryLevel(currMemLevel--);
	}
      } else if ((*comp->obj).value.func == f_repeat) {
	if (flow[nFlow].endValue == 0) {
	  skipProgram(&comp, f_end, NULL);
	  --nFlow;
	}
      } else if ((*comp->obj).value.func == f_end) {
	switch (flow[nFlow].type) {
	case CTR_IF:
	  --nFlow;
	  break;

	case CTR_CASE:
	  if (flow[nFlow].currValue == 0)
	    --nFlow;
	  else if (flow[nFlow].endValue == 1) {
	    comp = comp->next;
	    skipToEndOfCase(&comp);
	    --nFlow;
	  }
	  break;

	case CTR_WHILE:
	  comp = flow[nFlow].address;
	  break;

	case CTR_DO:
	  if (flow[nFlow].endValue == 0)
	    comp = flow[nFlow].address;
	  else
	    nFlow--;
	  break;

	default:
	  doError("end", ERR_SYNTAXERROR);
	}
      }
    }  
    
    comp = comp->next;
  }
    

  --progNesting;

  while (currMemLevel > memLevelAtEntry)
    clearMemoryLevel(currMemLevel--);
}


void
skipProgram(Composite **comp, void (*func1)(void), void (*func2)(void))
{
  /*
   * Skips the program until either of the given functions are found, or
   * until its end.
   */

  while (1) {
    if (!*comp)
      return;
    
    if ((*comp)->obj->type == TYPE_FUNC) {
      if ((*comp)->obj->value.func == func1
	  || (*comp)->obj->value.func == func2)
	return;
      if ((*comp)->obj->value.func == f_if
	  || (*comp)->obj->value.func == f_while
	  || (*comp)->obj->value.func == f_do) {
	*comp = (*comp)->next;
	skipProgram(comp, f_end, NULL);
	if ((*comp) == NULL)
	  doError("if/while/do", ERR_SYNTAXERROR);
      } else if ((*comp)->obj->value.func == f_case) {
	*comp = (*comp)->next;
	skipToEndOfCase(comp);
      } else if ((*comp)->obj->value.func == f_for
		 || (*comp)->obj->value.func == f_start) {
	*comp = (*comp)->next;
	skipProgram(comp, f_next, f_step);
	if (*comp == NULL)
	  doError("for/start", ERR_SYNTAXERROR);
      }
    }

    *comp = (*comp)->next;
  }
}


void
skipToEndOfCase(Composite **comp)
{
  /*
   * Skips till the end of the a CASE strucure.
   */

  while (1) {
    skipProgram(comp, f_then, f_end);
    if (*comp == NULL || (*comp)->obj->value.func == f_end)
      return;

    if ((*comp)->obj->value.func == f_then) {
      skipProgram(comp, f_end, NULL);
      if (*comp == NULL)
	return;
      *comp = (*comp)->next;
    }
  }
}


void
f_ift(void)
{
  /*
   * If the object in level 2 is a non-zero real-number, evaluates the object
   * in level one, otherwise drops the object in level one.
   */

  if (enoughArgs(2)) {
    if (type(**(tos - 1)) == TYPE_REAL) {
      saveLastArgs(2);

      _f_swap();

      if ((**tos).value.real) {
	_f_drop();
	f_eval();
      } else {
	_f_drop();
	_f_drop();
      }
    } else
      doError("ift", ERR_BADARGUMENTTYPE);
  } else
    doError("ift", ERR_TOOFEWARGUMENTS);
}


void
f_ifte(void)
{
  /*
   * If the object in level 3 is a non-zero real-number, evaluates the object
   * in level two, otherwise executes the object in level one.
   */

  if (enoughArgs(3)) {
    if (type(**(tos - 2)) == TYPE_REAL) {
      saveLastArgs(3);

      _f_roll(3, "ifte");

      if ((**tos).value.real) {
	_f_drop();
	_f_drop();
	f_eval();
      } else {
	_f_drop();
	_f_swap();
	_f_drop();
	f_eval();
      }
    } else
      doError("ifte", ERR_BADARGUMENTTYPE);
  } else
    doError("ifte", ERR_TOOFEWARGUMENTS);
}


void
f_dobind(void)
{
  /*
   * dobind function stack diagram:
   * obj1 ... objn id1 ... idn %n -->
   *
   * Creates a new temporary environment and binds n objects to n names.
   */

  int n;
  register int i;

  if (progNesting == 0)
    doError("dobind", ERR_NOTINPROGRAM);

  if (!enoughArgs(1))
    doError("dobind", ERR_TOOFEWARGUMENTS);

  if (type(**tos) != TYPE_REAL)
    doError("dobind", ERR_BADARGUMENTTYPE);

  /* Start the next memory level */
  initNextMemoryLevel();

  /* Get number of objects */
  n = (int) (**tos).value.real;
  if (!enoughArgs(2*n + 1))
    doError("dobind", ERR_TOOFEWARGUMENTS);
  if (n < 0)
    doError("dobind", ERR_BADARGUMENTVALUE);

  /* Check if all are IDs */
  for (i = 1; i <= n; i++)
    if (type(**(tos - i)) != TYPE_ID)
      doError("dobind", ERR_BADARGUMENTTYPE);

  _f_drop();      /* Remove number from stack */
  
  /* Do the binding */
  for (i = 0; i < n; i++)
    bindObject((**(tos - i)).value.str, *(tos - i - n));

  /* And remove them from the stack */
  tos -= 2*n;
}


void
f_abnd(void)
{
  /*
   * abnd function. Abandons topmost temporary environment.
   */

  if (progNesting == 0)
    doError("abnd", ERR_NOTINPROGRAM);

  if (currMemLevel == 0)
    doError("abnd", ERR_NOTEMPENVS);

  clearMemoryLevel(currMemLevel--);
}


void
f_arrow(void)
{
  /*
   * -> function.
   *
   * Stores the program nesting level at which the arrow was found,
   * and the depth. When << is found later, that information will be used.
   */

  if (progNesting == 0)
    doError("->", ERR_NOTINPROGRAM);

  if (nArrows == COMPLEVELS - 1)
    doError("->", ERR_RECURSIONTOODEEP);

  lastArrowLevel[++nArrows] = progNesting;
  lastArrowDepth[nArrows] = _f_depth();
}


void
f_if(void) {
  /*
   * This function stores information about the current IF construct.
   */

  if (progNesting == 0)
    doError("if", ERR_NOTINPROGRAM);

  if (nFlow == COMPLEVELS - 1)
    doError("if", ERR_RECURSIONTOODEEP);

  flow[++nFlow].type = CTR_IF;
  flow[nFlow].currValue = progNesting;
}


void
f_then(void) {
  /*
   * Checks if the THEN is valid, the actual execution is in
   * evalProgram().
   */

  if (nFlow == -1)
    doError("then", ERR_SYNTAXERROR);

  if (flow[nFlow].type == CTR_IF) {
    if (flow[nFlow].currValue != progNesting)
      doError("then", ERR_SYNTAXERROR);
  } else if (flow[nFlow].type != CTR_CASE)
    doError("then", ERR_SYNTAXERROR);
  
  if (!enoughArgs(1))
    doError("then", ERR_TOOFEWARGUMENTS);

  if (type(**tos) != TYPE_REAL)
    doError("then", ERR_BADARGUMENTTYPE);

  flow[nFlow].endValue = (**tos).value.real;
  _f_drop();
  if (flow[nFlow].type == CTR_CASE)
    flow[nFlow].currValue = 1;
}


void
f_else(void) {
  /*
   * Checks if the ELSE is valid, the actual execution is in
   * evalProgram().
   */

  if (nFlow == -1
      || flow[nFlow].type != CTR_IF
      || flow[nFlow].currValue != progNesting)
    doError("else", ERR_SYNTAXERROR);
}


void f_end(void) {
  /*
   * Checks if the END is valid.
   */

  if (nFlow == -1)
    doError("end", ERR_SYNTAXERROR);

  switch (flow[nFlow].type) {
  case CTR_IF:
  case CTR_CASE:
  case CTR_WHILE:
    break;

  case CTR_DO:
    if (!enoughArgs(1))
      doError("end", ERR_TOOFEWARGUMENTS);
    if (type(**tos) != TYPE_REAL)
      doError("end", ERR_BADARGUMENTTYPE);

    flow[nFlow].endValue = (**tos).value.real;
    _f_drop();
    break;

  default:
    doError("end", ERR_SYNTAXERROR);
  }
}


void
f_case(void)
{
  /*
   * Stores information about the CASE structure.
   * currValue indicates whether a THEN without its END has been found,
   * and endValue is the result of the last THEN.
   */

  if (progNesting == 0)
    doError("case", ERR_NOTINPROGRAM);

  if (nFlow == COMPLEVELS - 1)
    doError("case", ERR_RECURSIONTOODEEP);

  flow[++nFlow].type = CTR_CASE;
  flow[nFlow].currValue = 0;
  flow[nFlow].endValue = 0;
}


void
f_start(void)
{
  /*
   * Stores information about the current START structure.
   */

  if (progNesting == 0)
    doError("start", ERR_NOTINPROGRAM);

  if (nFlow == COMPLEVELS - 1)
    doError("start", ERR_RECURSIONTOODEEP);

  if (!enoughArgs(2))
    doError("start", ERR_TOOFEWARGUMENTS);

  if (type(**tos) != TYPE_REAL || type(**(tos - 1)) != TYPE_REAL)
    doError("start", ERR_BADARGUMENTTYPE);

  flow[++nFlow].type = CTR_DEFLOOP;
  flow[nFlow].ctrVarName = NULL;
  flow[nFlow].currValue = (**(tos - 1)).value.real;
  flow[nFlow].endValue = (**tos).value.real;
  _f_drop();
  _f_drop();
}


void
f_for(void)
{
  /*
   * The saving of the counter variable name is done in evalProgram().
   */

  f_start();
}


void
f_next(void)
{
  /*
   * Increments current loop count by 1.
   */

  incrLoop(1);
}


void
f_step(void)
{
  /*
   * Increments current loop count by the number in the stack.
   */

  double n;

  if (!enoughArgs(1))
    doError("step", ERR_TOOFEWARGUMENTS);

  if (type(**tos) != TYPE_REAL)
    doError("step", ERR_BADARGUMENTTYPE);

  n = (**tos).value.real;
  _f_drop();
  
  incrLoop(n);
}


void
incrLoop(double n)
{
  /*
   * If applicable, increments the last loop structure's count by the
   * given amount.
   */

  if (nFlow == -1 || flow[nFlow].type != CTR_DEFLOOP)
    doError("step/next", ERR_SYNTAXERROR);

  flow[nFlow].currValue += n;

  if (flow[nFlow].ctrVarName)
    (*recallObject(flow[nFlow].ctrVarName)).value.real = flow[nFlow].currValue;
}


void
f_while(void)
{
  /*
   * Marks the beginning of a while loop.
   */

  if (progNesting == 0)
    doError("while", ERR_NOTINPROGRAM);

  if (nFlow == COMPLEVELS - 1)
    doError("while", ERR_RECURSIONTOODEEP);

  flow[++nFlow].type = CTR_WHILE;
}


void
f_repeat(void)
{
  /*
   * Checks if execution should continue.
   */

  if (nFlow == -1 || flow[nFlow].type != CTR_WHILE)
    doError("repeat", ERR_SYNTAXERROR);

  if (!enoughArgs(1))
    doError("repeat", ERR_TOOFEWARGUMENTS);

  if (type(**tos) != TYPE_REAL)
    doError("repeat", ERR_BADARGUMENTTYPE);

  flow[nFlow].endValue = (**tos).value.real;
  _f_drop();
}


void
f_do(void)
{
  /*
   * Marks beginning of do loop.
   */

  if (progNesting == 0)
    doError("do", ERR_NOTINPROGRAM);

  if (nFlow == COMPLEVELS - 1)
    doError("do", ERR_RECURSIONTOODEEP);

  flow[++nFlow].type = CTR_DO;
}


void
f_until(void)
{
  /*
   * Just checks if its presence is valid.
   */

  if (nFlow == -1 || flow[nFlow].type != CTR_DO)
    doError("until", ERR_SYNTAXERROR);
}
